/* ===== Utils: tracking ===== */
window.revo = window.revo || {};
revo.track = function(eventName, params = {}) {
  // Console fallback si GA4 absent
  if (typeof window.gtag !== "function") {
    console.log("[track]", eventName, params);
    return;
  }
  window.gtag("event", eventName, params);
};

function bindTracking() {
  // Clics marqués data-evt
  document.querySelectorAll("[data-evt]").forEach(el => {
    el.addEventListener("click", () => {
      const name = el.getAttribute("data-evt");
      const ctx = el.getAttribute("data-ctx") || "ui";
      const url = el.getAttribute("data-url") || (el.href || "");
      revo.track(name, { context: ctx, url });
    });
  });

  // Outbound link (fallback)
  document.body.addEventListener("click", (e) => {
    const a = e.target.closest("a[href]");
    if (!a) return;
    const url = a.getAttribute("href");
    try {
      const host = new URL(url, location.href).host;
      if (host && host !== location.host) {
        revo.track("outbound_click", { url });
      }
    } catch(_) {}
  });
}

/* ===== Header: mobile menu ===== */
function setupMenu() {
  const toggle = document.querySelector(".nav__toggle");
  const menu = document.getElementById("nav-menu") || document.querySelector(".nav__menu");
  if (!toggle || !menu) return;

  const mq = window.matchMedia("(max-width:1023px)");

  function open()  {
    menu.classList.add("is-open");
    toggle.setAttribute("aria-expanded", "true");
    document.addEventListener("keydown", onKey);
  }
  function close() {
    menu.classList.remove("is-open");
    toggle.setAttribute("aria-expanded", "false");
    document.removeEventListener("keydown", onKey);
  }
  function onKey(ev){ if (ev.key === "Escape") close(); }

  toggle.addEventListener("click", (e) => {
    e.preventDefault();
    menu.classList.contains("is-open") ? close() : open();
  });

  // Fermer le menu après clic sur un lien **uniquement en mobile**
  menu.addEventListener("click", (e) => {
    const a = e.target.closest("a");
    if (!a) return;
    if (mq.matches) close();
  });

  // En repassant en desktop, on garantit l'état propre
  mq.addEventListener("change", (ev) => {
    if (!ev.matches) {
      menu.classList.remove("is-open");
      toggle.setAttribute("aria-expanded", "false");
    }
  });
}

/* ===== Modal example ===== */
function setupModal() {
  const modal = document.getElementById("example-modal");
  if (!modal) return;
  const closeBtns = modal.querySelectorAll(".js-close-modal");
  function close() {
    modal.hidden = true;
    document.removeEventListener("keydown", onKey);
  }
  function onKey(ev){ if (ev.key === "Escape") close(); }
  closeBtns.forEach(btn => btn.addEventListener("click", close));
  modal.addEventListener("click", (e)=>{ if(e.target === modal) close(); });
}

/* ===== Form validation (accessible) ===== */
function setupForm() {
  const form = document.getElementById("contact-form");
  if (!form) return;

  const fields = {
    name: document.getElementById("name"),
    email: document.getElementById("email"),
    org: document.getElementById("org"),
    objectif: document.getElementById("objectif"),
    consent: document.getElementById("consent"),
  };
  const errors = {
    name: document.getElementById("err-name"),
    email: document.getElementById("err-email"),
    org: document.getElementById("err-org"),
    objectif: document.getElementById("err-objectif"),
  };
  const status = document.getElementById("form-status");

  let started = false;
  form.addEventListener("focusin", () => {
    if (!started) {
      revo.track("form_started", { context: "contact" });
      started = true;
    }
  });

  function setError(el, errEl, msg) {
    errEl.textContent = msg || "";
    if (msg) el.setAttribute("aria-invalid", "true");
    else el.removeAttribute("aria-invalid");
  }

  function validate() {
    let ok = true;
    if (!fields.name.value.trim()) { setError(fields.name, errors.name, "Nom requis"); ok = false; } else setError(fields.name, errors.name, "");
    const emailVal = fields.email.value.trim();
    const emailOk = /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(emailVal);
    if (!emailOk) { setError(fields.email, errors.email, "Email valide requis"); ok = false; } else setError(fields.email, errors.email, "");
    if (!fields.org.value.trim()) { setError(fields.org, errors.org, "Organisation requise"); ok = false; } else setError(fields.org, errors.org, "");
    if (!fields.objectif.value) { setError(fields.objectif, errors.objectif, "Sélection requise"); ok = false; } else setError(fields.objectif, errors.objectif, "");
    if (!fields.consent.checked) { ok = false; alert("Veuillez accepter la politique de protection des données."); }
    return ok;
  }

  form.addEventListener("submit", (e) => {
    e.preventDefault();
    if (!validate()) return;
    // 👉 brancher votre backend / Make ici
    status.textContent = "Merci ! Nous revenons vers vous très vite.";
    revo.track("demo_requested", { context: "contact_form" });
    form.reset();
  });
}

/* ===== Lazy load Stripe pricing table ===== */
function setupStripeLazy() {
  const box = document.getElementById("stripe-pricing");
  if (!box) return;
  const tableId = box.dataset.pricingTableId;
  const pubKey = box.dataset.pubKey;

  const loadStripe = () => {
    if (box.dataset.loaded) return;
    box.dataset.loaded = "1";
    const s = document.createElement("script");
    s.src = "https://js.stripe.com/v3/pricing-table.js";
    s.async = true;
    s.onload = () => {
      const el = document.createElement("stripe-pricing-table");
      el.setAttribute("pricing-table-id", tableId);
      el.setAttribute("publishable-key", pubKey);
      box.appendChild(el);
      revo.track("stripe_table_loaded");
    };
    document.head.appendChild(s);
  };

  const io = new IntersectionObserver((entries) => {
    entries.forEach(e => { if (e.isIntersecting) { loadStripe(); io.disconnect(); } });
  }, { threshold: 0.15 });
  io.observe(box);
}

/* ===== Cookie consent (analytics) ===== */
function setupConsent() {
  const banner = document.getElementById("cookie-banner");
  if (!banner) return; // pas de bannière → on sort proprement

  const CONSENT_KEY = "revo_cookie_consent";
  const saved = localStorage.getItem(CONSENT_KEY);

  // Supporte 2 syntaxes d'attributs dans le HTML
  const acceptBtn = banner.querySelector("[data-cookie-accept], [data-consent='grant']");
  const denyBtn   = banner.querySelector("[data-cookie-deny], [data-consent='deny']");

  const show = () => { banner.hidden = false; };
  const hide = () => { banner.hidden = true; };

  function updateGtag(value) {
    if (typeof window.gtag === "function") {
      const v = value === "granted" ? "granted" : "denied";
      gtag("consent", "update", {
        ad_storage: v,
        analytics_storage: v,
        ad_user_data: v,
        ad_personalization: v
      });
    }
  }

  function setConsent(value) {
    localStorage.setItem(CONSENT_KEY, value);
    updateGtag(value);
    revo.track(value === "granted" ? "consent_granted" : "consent_denied");
    hide();
  }

  if (!saved) show(); // pas de choix → on affiche
  else updateGtag(saved); // si déjà choisi, on aligne GA4

  acceptBtn?.addEventListener("click", () => setConsent("granted"));
  denyBtn?.addEventListener("click",   () => setConsent("denied"));
}

/* ===== Init ===== */
document.addEventListener("DOMContentLoaded", () => {
  const y = document.getElementById("year");
  if (y) y.textContent = new Date().getFullYear();

  bindTracking();
  setupMenu();
  setupModal();
  setupForm();
  setupStripeLazy();
  setupConsent();
});

